#!/bin/bash
#
# Backup script
#
# Copyright Daniel Mehrmann (c) 2018
# Created 09.12.18
#
# 11.12.2018: bugfix: forgotten /boot partition inside backup
# 18.01.2019: Changes for Homer-Mobil partion layout
# 26.04.2019: Check if directory /backup/Homer-Mobil exist
# 11.01.2021: Add pigz for multi thread compression
# 07.03.2022: Add $workind_dir to use realative path 
# 07.03.2023: Changes for daten-box lvm configuration
# 21.05.2023: Fix device for boot partition
# 16.06.2023: for /boot switch to UUID because changing dev id between nvme's
# 25.09.2023: Copy backup to /var/local/pool/backup/backup/daten-box-copy
# 21.01.2024: Add $hostname
# 21.01.2024: Add $1 systemd for timer/cronjob staff
# 02.03.2024: Add $boot_device
# 17.03.2024: New backup host/share handling.
# 17.03.2024: Add DB backup script for nextcloud

### configuration ###

date="$(/usr/bin/date "+%F")"

if [ -f /usr/local/etc/make_backup.conf ]; then
   source /usr/local/etc/make_backup.conf || { echo "Can't read configuration file!"; exit 255; }
else
   echo "/usr/local/make_backup.conf doesn't exist!"
   exit 255
fi

if [ "$new_backup" == "yes" ]; then
   backupfile="/backup/system-${date}.tar.gz"
else
   backupfile="/backup/${hostname}/system-${date}.tar.gz"
fi
working_dir="/usr/local/sbin"

### switch to working dir to use relative path
cd "$working_dir" || { echo "Can't switch to working dir"; exit 255; }

### prepare /backup ###

/usr/bin/mount /backup || { echo "Can't mount /backup"; exit 255; }

if [[ "$new_backup" == "no" && ! -d "/backup/${hostname}" ]]; then
	mkdir -p /backup/"${hostname}" || { echo "Can' create /backup/${hostname}"; exit 255; }
fi

### create and mount snapshots under /mnt ###

for i in lvm_pre_*; do
   if [ "$1" == "boot" ]; then
      if [ "$i" == "lvm_pre_0_snap_make.sh" ]; then
         continue;
      fi
   fi
   ret=$(exec $i) 
   if [ "$ret" == "255" ]; then
      echo "Fatal error on $i"
      exit 255
   fi
done

### Unmount and mount /boot under /mnt/boot ###

/usr/bin/umount /boot || { echo "Can't unmount /boot"; exit 255; }
/usr/bin/mount -o ro $boot_device /mnt/boot || { echo "Can't mount boot under /mnt/boot"; exit 255; }

### Do backup ###

if [ "$1" == "systemd" ]; then
   output=""
else
   output="-v"
fi

echo "Starting backup..."

if [ "$use_pigz" == "yes" ]; then
	/usr/bin/tar --xattrs -cp  $output -C /mnt . |pigz -p $use_pigz_threads > "$backupfile"
else
	/usr/bin/tar --xattrs -czp $output -f "$backupfile" -C /mnt .
fi

### Unmount and mount boot back to /boot ###

/usr/bin/umount /mnt/boot || { echo "Can't unmount /backup/mnt/boot"; exit 255; }
/usr/bin/mount /boot || { echo "Can't mount boot back to /boot"; exit 255; } 

### cleanup ###

for i in lvm_post_*; do
   ret=$(exec $i) 
   if [ "$ret" == "255" ]; then
      echo "Fatal error on $i"
      exit 255
   fi
done

### Copy backup ###

if [ "$hostname" == "daten-box" ]; then
   cp "$backupfile" /var/local/pool/backup/backup/daten-box-copy/daten-box-latest.tar.gz 
fi

### DB backup ###
if [ "$hostname" == "nextcloud" ]; then
   /usr/local/sbin/make_db_dump.sh /backup
fi

/usr/bin/umount /backup

echo "Everything runs smoothly :-)"
exit 0 
