#!/bin/bash
#
# Backup script
#
# Copyright Daniel Mehrmann (c) 2018
# Created 09.12.18
#
# 11.12.2018: bugfix: forgotten /boot partition inside backup
# 18.01.2019: Changes for Homer-Mobil partion layout
# 26.04.2019: Check if directory /backup/Homer-Mobil exist
# 11.01.2021: Add pigz for multi thread compression
# 07.03.2022: Add $workind_dir to use realative path 
# 07.03.2023: Changes for daten-box lvm configuration
# 21.05.2023: Fix device for boot partition
# 16.06.2023: for /boot switch to UUID because changing dev id between nvme's
# 25.09.2023: Copy backup to /var/local/pool/backup/backup/daten-box-copy
# 21.01.2024: Add $hostname
# 21.01.2024: Add $1 systemd for timer/cronjob staff
# 02.03.2024: Add $boot_device
# 17.03.2024: New backup host/share handling.
# 17.03.2024: Add DB backup script for nextcloud
# 30.12.2024: Fix design error to hanfle snapshots more safty 

### configuration ###

date="$(/usr/bin/date "+%F")"
failed_ret=0

if [ -f /usr/local/etc/make_backup.conf ]; then
   source /usr/local/etc/make_backup.conf || { echo "Can't read configuration file!"; exit 255; }
else
   echo "/usr/local/make_backup.conf doesn't exist!"
   exit 255
fi

# shellcheck disable=SC2154
if [ "$new_backup" == "yes" ]; then
   backupfile="/backup/system-${date}.tar.gz"
else
   # shellcheck disable=SC2154
   backupfile="/backup/${hostname}/system-${date}.tar.gz"
fi
working_dir="/usr/local/sbin"

### check conditions
if [ -f /.backup_run ]; then
   echo "Lockfile from previous backup found! Aborting..."
   exit 255
fi

### switch to working dir to use relative path
cd "$working_dir" || { echo "Can't switch to working dir"; exit 255; }

### prepare /backup ###

/usr/bin/mount /backup || { echo "Can't mount /backup"; exit 255; }

if [[ "$new_backup" == "no" && ! -d "/backup/${hostname}" ]]; then
	mkdir -p /backup/"${hostname}" || { echo "Can' create /backup/${hostname}"; exit 255; }
fi

### create and mount snapshots under /mnt ###

# create lock file
touch /.backup_run || exit 255

for i in lvm_pre_*; do
   if [ "$1" == "boot" ]; then
      if [ "$i" == "lvm_pre_0_snap_make.sh" ]; then
         continue;
      fi
   fi
   ret=$(exec $i) 
   if [ "$ret" == "255" ]; then
      echo "Fatal error on $i"
      failed_ret=255 
   fi
done

# error handling
if [ $failed_ret == 255 ]; then
   ret=$(exec lvm_post_1_delete.sh)
   if [ "$ret" == 255 ]; then
      echo "$(basename "$0"): Somewhat with LVM snapshots aren't okay! Please check it." | mail Ssendwait -s "$(cat /etc/hostname) WARNING: make_backup.sh" root@localhost       
     
      echo -e '\a' > /dev/console
      sleep 2
      echo -e '\a' > /dev/console
      sleep 2
      echo -e '\a' > /dev/console
      sleep 2
      echo -e '\a' > /dev/console
   fi
fi   

### Unmount and mount /boot under /mnt/boot ###

/usr/bin/umount /boot || { echo "Can't unmount /boot"; failed_ret=255; }
# shellcheck disable=SC2154
/usr/bin/mount -o ro "$boot_device" /mnt/boot || { echo "Can't mount boot under /mnt/boot"; failed_ret=255; }

### Do backup ###

if [ "$1" == "systemd" ]; then
   output=""
else
   output="-v"
fi

if  [ $failed_ret != 255 ]; then
   echo "Starting backup..."

   # shellcheck disable=SC2154
   if [ "$use_pigz" == "yes" ]; then
      # shellcheck disable=SC2154
      /usr/bin/tar --xattrs -cp  $output -C /mnt . |pigz -p "$use_pigz_threads" > "$backupfile" || failed_ret=$?
   else
      /usr/bin/tar --xattrs -czp $output -f "$backupfile" -C /mnt . || failed_ret=$?
   fi
fi

### Unmount and mount boot back to /boot ###

/usr/bin/umount /mnt/boot || { echo "Can't unmount /backup/mnt/boot"; failed_ret=255; }
/usr/bin/mount /boot || { echo "Can't mount boot back to /boot"; failed_ret=255; } 

### cleanup ###

for i in lvm_post_*; do
   ret=$(exec $i) 
   if [ "$ret" == "255" ]; then
      echo "Fatal error on $i"
      failed_ret=255
   fi
done

# error handling
if [ $failed_ret == 255 ]; then
   ret=$(exec lvm_post_1_delete.sh)
   if [ "$ret" == 255 ]; then
      echo "$(basename "$0"): Somewhat with LVM snapshots aren't okay! Please check it." | mail Ssendwait -s "$(cat /etc/hostname) WARNING: make_backup.sh" root@localhost

      echo -e '\a' > /dev/console
      sleep 2
      echo -e '\a' > /dev/console
      sleep 2
      echo -e '\a' > /dev/console
      sleep 2
      echo -e '\a' > /dev/console
   fi
fi

### Copy backup ###

if [ "$hostname" == "daten-box" ]; then
   cp "$backupfile" /var/local/pool/backup/backup/daten-box-copy/daten-box-latest.tar.gz 
fi

### DB backup ###
if [ "$hostname" == "nextcloud" ]; then
   /usr/local/sbin/make_db_dump.sh /backup
fi

# clean up work
/usr/bin/umount /backup || exit 255
rm /.backup_run || exit 255

echo "Everything runs smoothly :-)"

exit $failed_ret 
